<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Employee QR Code Scanner</title>
    <style>
        #decoding-style {
            display: none;
        }
        input[type="submit"][name="next"] {
            display: none;
        }
        #employeeDetails {
            display: none;
            text-align: center;
            position: absolute;
            top: 20px;
            left: 50%;
            transform: translateX(-50%);
            background-color: #ffffff;
            padding: 20px;
            border-radius: 10px;
        }
        #employeeDetails img {
            max-width: 100px;
            border-radius: 50%;
        }
        #countdown {
            display: none;
            font-size: 2em;
            text-align: center;
            margin-top: 20px;
        }
        #errorMessage {
            display: none;
            color: red;
            text-align: center;
            margin-top: 20px;
        }
        #faceDetectionContainer {
            display: none;
            text-align: center;
        }
        #faceVideo, #overlay {
            border: 1px solid #ccc;
        }
    </style>
</head>
<body>
    <form id="qrCodeForm" action="qrcheck.php" method="post">
        <div id="employeeDetails">
            <h2 id="employeeName"></h2>
            <img id="employeePhoto" src="" alt="Employee Photo">
        </div>
        <a id="startButton">
            <video id="video" width="100%" height="350" poster="scan.png"></video>
        </a>
        <input type="hidden" id="employee" name="employee" class="employee-input">
        <select id="decoding-style">
            <option value="once"></option>
        </select>
        <input type="submit" name="next" value="Next" id="nextButton">
        <div id="countdown">5</div>
        <div id="errorMessage">Error: Could not access the camera. Please check your camera permissions and try again.</div>
    </form>
    <div id="faceDetectionContainer">
        <video id="faceVideo" width="720" height="560" autoplay muted></video>
        <canvas id="overlay" width="720" height="560"></canvas>
    </div>

    <script type="text/javascript">
        let qrCodeDetected = false;
        let faceDetected = false;
        let countdownInterval;

        function showError(message) {
            const errorMessage = document.getElementById('errorMessage');
            errorMessage.textContent = message;
            errorMessage.style.display = 'block';
        }

        function decodeOnce(codeReader, selectedDeviceId) {
            codeReader.decodeFromInputVideoDevice(selectedDeviceId, 'video').then((result) => {
                if (result && !qrCodeDetected) {
                    console.log('Found QR code!', result);
                    qrCodeDetected = true;
                    document.getElementById('employee').value = result.text.trim();
                    fetchEmployeeDetails(result.text.trim());
                }
            }).catch((err) => {
                console.error(err);
                if (err instanceof ZXing.NotFoundException) {
                    console.log('No QR code found.');
                } else {
                    showError('Error reading QR code: ' + err.message);
                }
            });
        }

        function fetchEmployeeDetails(employeeId) {
            fetch('getEmployeeDetails.php?employee=' + employeeId)
                .then(response => response.json())
                .then(data => {
                    if (data.error) {
                        alert(data.message);
                    } else {
                        document.getElementById('employeeName').textContent = 'Welcome ' + data.firstname + ' ' + data.lastname;
                        document.getElementById('employeePhoto').src = 'images/' + data.photo;
                        document.getElementById('employeeDetails').style.display = 'block';
                        document.getElementById('video').style.display = 'none';
                        document.getElementById('faceDetectionContainer').style.display = 'block';
                        initializeFaceDetection();
                    }
                })
                .catch(error => console.error('Error fetching employee details:', error));
        }

        function initializeFaceDetection() {
            const videoElement = document.getElementById('faceVideo');
            const canvasElement = document.getElementById('overlay');
            const canvasCtx = canvasElement.getContext('2d');

            async function setupCamera() {
                try {
                    const stream = await navigator.mediaDevices.getUserMedia({ video: true });
                    videoElement.srcObject = stream;
                    return new Promise(resolve => {
                        videoElement.onloadedmetadata = () => {
                            resolve(videoElement);
                        };
                    });
                } catch (error) {
                    console.error('Error accessing webcam:', error);
                }
            }

            function onResults(results) {
                canvasCtx.save();
                canvasCtx.clearRect(0, 0, canvasElement.width, canvasElement.height);
                canvasCtx.drawImage(results.image, 0, 0, canvasElement.width, canvasElement.height);
                if (results.detections) {
                    results.detections.forEach(detection => {
                        window.drawLandmarks(canvasCtx, detection.landmarks, { color: 'red', lineWidth: 2 });
                        window.drawRectangle(canvasCtx, detection.boundingBox, { color: 'red', lineWidth: 1 });
                    });
                }
                canvasCtx.restore();

                if (results.detections && results.detections.length > 0) {
                    if (!faceDetected) {
                        faceDetected = true;
                        startCountdown();
                    }
                } else {
                    faceDetected = false;
                    resetCountdown();
                }
            }

            async function start() {
                await setupCamera();
                const faceDetection = new window.FaceDetection({
                    locateFile: file => `https://cdn.jsdelivr.net/npm/@mediapipe/face_detection/${file}`
                });
                faceDetection.setOptions({ model: 'short', minDetectionConfidence: 0.5 });
                faceDetection.onResults(onResults);

                async function detect() {
                    await faceDetection.send({ image: videoElement });
                    requestAnimationFrame(detect);
                }

                videoElement.play();
                detect();
            }

            start();
        }

        function startCountdown() {
            let countdown = 5;
            document.getElementById('countdown').style.display = 'block';
            countdownInterval = setInterval(() => {
                document.getElementById('countdown').textContent = countdown;
                if (countdown === 0) {
                    clearInterval(countdownInterval);
                    document.getElementById('qrCodeForm').submit();
                }
                countdown--;
            }, 1000);
        }

        function resetCountdown() {
            clearInterval(countdownInterval);
            document.getElementById('countdown').style.display = 'none';
            document.getElementById('countdown').textContent = '5';
        }

        window.addEventListener('load', function () {
            let selectedDeviceId;
            const codeReader = new ZXing.BrowserQRCodeReader();
            console.log('ZXing code reader initialized');

            codeReader.getVideoInputDevices()
                .then((videoInputDevices) => {
                    const frontCameraDevices = videoInputDevices.filter(device => device.label.toLowerCase().includes('front') || device.facingMode === 'user');

                    if (frontCameraDevices.length > 0) {
                        selectedDeviceId = frontCameraDevices[0].deviceId;
                        console.log(`Using front camera with id ${selectedDeviceId}`);
                    } else {
                        showError('No front camera found. Please ensure your device has a front camera.');
                    }

                    document.getElementById('startButton').addEventListener('click', () => {
                        const decodingStyle = document.getElementById('decoding-style').value;

                        if (decodingStyle === "once") {
                            decodeOnce(codeReader, selectedDeviceId);
                        }
                    });
                })
                .catch((err) => {
                    console.error('Error getting video input devices:', err);
                    showError('Error: Could not access the camera. Please check your camera permissions and try again.');
                });
        });
    </script>
    <script src="https://cdn.jsdelivr.net/npm/@mediapipe/face_detection/face_detection.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/@mediapipe/drawing_utils/drawing_utils.js"></script>
    <script type="text/javascript" src="js/qrscan.js"></script>
</body>
</html>
